<?php
session_start();
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../auth/auth.php';

// Set proper headers
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Not authorized']);
    exit;
}

try {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // If no JSON input, try POST data
    if (!$input) {
        $input = $_POST;
    }

    // Log the received data for debugging
    error_log("Received refund data: " . print_r($input, true));

    // Validate input
    if (!isset($input['sale_id']) || !isset($input['refund_amount']) || !isset($input['refund_reason'])) {
        throw new Exception('Missing required fields');
    }

    $sale_id = $input['sale_id'];
    $refund_amount = floatval($input['refund_amount']);
    $refund_reason = $input['refund_reason'];
    $refund_items = $input['refund_items'];
    $user_id = $_SESSION['user_id'];

    // Start transaction
    $pdo->beginTransaction();

    // Get sale details
    $stmt = $pdo->prepare("
        SELECT 
            s.*,
            c.name as customer_name,
            ct.id as credit_transaction_id,
            ct.amount as credit_amount
        FROM sales s
        LEFT JOIN customers c ON s.customer_id = c.id
        LEFT JOIN credit_transactions ct ON s.id = ct.sale_id
        WHERE s.id = ?
    ");
    $stmt->execute([$sale_id]);
    $sale = $stmt->fetch();

    if (!$sale) {
        throw new Exception('Sale not found');
    }

    // Validate refund amount
    if ($refund_amount <= 0) {
        throw new Exception('Refund amount must be greater than 0');
    }

    // For credit sales, check if refund amount doesn't exceed credit balance
    if ($sale['payment_method'] === 'credit' && $sale['credit_transaction_id']) {
        $stmt = $pdo->prepare("
            SELECT 
                ct.amount,
                COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid_amount,
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived_amount
            FROM credit_transactions ct
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            WHERE ct.id = ?
            GROUP BY ct.id, ct.amount
        ");
        $stmt->execute([$sale['credit_transaction_id']]);
        $credit_info = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($credit_info) {
            $credit_balance = $credit_info['amount'] - $credit_info['paid_amount'] - $credit_info['waived_amount'];
            
            // Check if all items being returned are in "Good" condition
            $allItemsGood = true;
            foreach ($refund_items as $item) {
                if (isset($item['condition']) && $item['condition'] !== 'Good') {
                    $allItemsGood = false;
                    break;
                }
            }
            
            if (!$allItemsGood && $refund_amount > $credit_balance) {
                throw new Exception("Refund amount cannot exceed credit balance of " . number_format($credit_balance, 2) . " for damaged/opened items.");
            }
            // For "Good" condition items, allow return regardless of credit balance (including 0.00)
            // The refund will create a credit balance or reduce existing balance
        }
    } else {
        // For non-credit sales, check against total amount
        if ($refund_amount > $sale['total_amount']) {
            throw new Exception('Refund amount cannot exceed sale total');
        }
    }

    // Generate refund receipt number
    $refund_receipt_no = 'REF-' . date('Ymd') . '-' . str_pad($sale_id, 4, '0', STR_PAD_LEFT);

    // Create refund record
    try {
        $stmt = $pdo->prepare("
            INSERT INTO refunds (
                sale_id,
                amount,
                reason,
                status,
                payment_method,
                refunded_at,
                processed_by,
                created_at,
                updated_at,
                refund_receipt_no
            ) VALUES (?, ?, ?, 'completed', ?, NOW(), ?, NOW(), NOW(), ?)
        ");
        
        error_log("Refunds insert params: " . json_encode([
            'sale_id' => $sale_id,
            'amount' => $refund_amount,
            'reason' => $refund_reason,
            'payment_method' => $sale['payment_method'],
            'user_id' => $user_id,
            'refund_receipt_no' => $refund_receipt_no
        ]));
        
        $stmt->execute([
            $sale_id,
            $refund_amount,
            $refund_reason,
            $sale['payment_method'],
            $user_id,
            $refund_receipt_no
        ]);
        $refund_id = $pdo->lastInsertId();
        
        error_log("Refunds insert successful with ID: " . $refund_id);
    } catch (Exception $e) {
        error_log("Refunds insert error: " . $e->getMessage());
        throw $e;
    }

    // Process refund items
    $restocked_items = 0;
    foreach ($refund_items as $item) {
        // Verify sale item exists
        $stmt = $pdo->prepare("SELECT id FROM sale_items WHERE sale_id = ? AND product_id = ?");
        $stmt->execute([$sale_id, $item['product_id']]);
        $sale_item = $stmt->fetch();
        
        if (!$sale_item) {
            throw new Exception('Sale item not found for product ID: ' . $item['product_id']);
        }

        // Update product quantity (return to stock)
        $stmt = $pdo->prepare("
            UPDATE products p
            JOIN sale_items si ON p.id = si.product_id
            SET p.quantity = p.quantity + ?
            WHERE si.sale_id = ? AND si.product_id = ?
        ");
        $stmt->execute([$item['quantity'], $sale_id, $item['product_id']]);
        $restocked_items += $item['quantity'];

        // Get serial numbers that were sold in this sale before returning them
        $returned_serials = [];
        if (!empty($item['selected_serials'])) {
            // Use the selected serials from the frontend
            $returned_serials = $item['selected_serials'];
            error_log("Using selected serials from frontend: " . implode(', ', $returned_serials));
        } else {
            // Fallback: get the first N serials (check both manual_adjustment and credit_sale source types)
            $stmt = $pdo->prepare("
                SELECT serial_number 
                FROM product_serial_numbers 
                WHERE product_id = ? 
                AND source_type IN ('manual_adjustment', 'credit_sale')
                AND source_id = ?
                AND status = 'sold'
                ORDER BY created_at ASC
                LIMIT ?
            ");
            $stmt->execute([$item['product_id'], $sale_id, $item['quantity']]);
            $returned_serials = $stmt->fetchAll(PDO::FETCH_COLUMN);
            error_log("Found serials from database for product " . $item['product_id'] . " in sale " . $sale_id . ": " . implode(', ', $returned_serials));
        }
        
        // Return serial numbers to available status
        if (!empty($returned_serials)) {
            $inClause = implode(',', array_fill(0, count($returned_serials), '?'));
            $stmt = $pdo->prepare("
                UPDATE product_serial_numbers 
                SET status = 'available',
                    source_type = 'refund',
                    source_id = ?,
                    updated_at = NOW()
                WHERE product_id = ? 
                AND source_type IN ('manual_adjustment', 'credit_sale')
                AND source_id = ?
                AND status = 'sold'
                AND serial_number IN ($inClause)
            ");
            $params = array_merge([$refund_id, $item['product_id'], $sale_id], $returned_serials);
            $stmt->execute($params);
            
            error_log("Returned " . count($returned_serials) . " serial numbers to available status for product " . $item['product_id'] . " in refund " . $refund_id . ": " . implode(', ', $returned_serials));
        } else {
            error_log("No serial numbers found to return for product " . $item['product_id'] . " in sale " . $sale_id);
        }

        // Record refund item
        $stmt = $pdo->prepare("
            INSERT INTO refund_items (
                refund_id,
                sale_item_id,
                product_id,
                quantity,
                unit_price,
                total_price,
                restock_quantity,
                discount,
                serial_numbers
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $serial_numbers_json = !empty($returned_serials) ? json_encode($returned_serials) : null;
        error_log("Saving serial numbers to refund_items: " . $serial_numbers_json);
        
        $stmt->execute([
            $refund_id,
            $sale_item['id'],
            $item['product_id'],
            $item['quantity'],
            $item['price'],
            $item['price'] * $item['quantity'],
            $item['quantity'],
            $item['discount'] ?? 0,
            $serial_numbers_json
        ]);
    }

    // Handle multiple payment methods for refunds
    $stmt = $pdo->prepare("SELECT method, amount FROM sale_payments WHERE sale_id = ? ORDER BY created_at ASC");
    $stmt->execute([$sale_id]);
    $sale_payments = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($sale_payments)) {
        // Fallback to single payment method from sales table
        $sale_payments = [[
            'method' => $sale['payment_method'],
            'amount' => $sale['total_amount']
        ]];
    }
    
    // Calculate refund amounts for each payment method proportionally
    // Only consider real payments (not credit debt) for proportional calculation
    $real_payments = array_filter($sale_payments, function($payment) {
        return strtolower($payment['method']) !== 'credit';
    });
    
    $total_real_paid = array_sum(array_column($real_payments, 'amount'));
    $refund_breakdown = [];
    
    foreach ($sale_payments as $payment) {
        $payment_method = strtolower($payment['method']);
        $payment_amount = floatval($payment['amount']);
        
        if ($payment_amount > 0) {
            if ($payment_method === 'credit') {
                // For credit, refund the full credit amount (up to refund amount)
                $credit_refund = min($payment_amount, $refund_amount);
                $refund_breakdown[] = [
                    'method' => $payment_method,
                    'original_amount' => $payment_amount,
                    'refund_amount' => $credit_refund,
                    'is_debt' => true
                ];
            } else {
                // For real payments, calculate proportional refund
                if ($total_real_paid > 0) {
                    $proportional_refund = ($payment_amount / $total_real_paid) * $refund_amount;
                    $refund_breakdown[] = [
                        'method' => $payment_method,
                        'original_amount' => $payment_amount,
                        'refund_amount' => $proportional_refund,
                        'is_debt' => false
                    ];
                }
            }
        }
    }
    
    // Process refunds for each payment method
    foreach ($refund_breakdown as $refund) {
        if ($refund['method'] === 'credit') {
            // Handle credit refund
            if ($sale['credit_transaction_id']) {
                try {
                    $stmt = $pdo->prepare("
                        INSERT INTO credit_payments (
                            credit_transaction_id,
                            sale_id,
                            customer_id,
                            amount,
                            payment_method,
                            reference,
                            created_by,
                            is_waived,
                            status
                        ) VALUES (?, ?, ?, ?, 'refund', ?, ?, 0, 'refunded')
                    ");
                    
                    error_log("Credit payment insert params: " . json_encode([
                        'credit_transaction_id' => $sale['credit_transaction_id'],
                        'sale_id' => $sale_id,
                        'customer_id' => $sale['customer_id'],
                        'amount' => $refund['refund_amount'],
                        'reference' => 'Refund for Sale #' . $sale_id,
                        'user_id' => $user_id
                    ]));
                    
                    $stmt->execute([
                        $sale['credit_transaction_id'],
                        $sale_id,
                        $sale['customer_id'],
                        $refund['refund_amount'],
                        'Refund for Sale #' . $sale_id,
                        $user_id
                    ]);
                    
                    error_log("Credit payment inserted successfully with ID: " . $pdo->lastInsertId());
                } catch (Exception $e) {
                    error_log("Credit payment insert error: " . $e->getMessage());
                    throw $e;
                }
            }
        } else {
            // Handle cash/mpesa/other payment method refunds
            // These are typically handled by returning cash or processing bank/mpesa refunds
            error_log("Non-credit refund processed: " . $refund['method'] . " - " . $refund['refund_amount']);
        }
    }

    // Check if this refund completes the sale (all items refunded)
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(DISTINCT si.id) as total_sale_items,
            SUM(si.quantity) as total_sale_quantity,
            COUNT(DISTINCT ri.sale_item_id) as refunded_sale_items,
            SUM(ri.quantity) as total_refunded_quantity
        FROM sale_items si
        LEFT JOIN refund_items ri ON si.id = ri.sale_item_id
        WHERE si.sale_id = ?
    ");
    $stmt->execute([$sale_id]);
    $refund_summary = $stmt->fetch();

    $is_complete_refund = false;
    if ($refund_summary) {
        $is_complete_refund = ($refund_summary['refunded_sale_items'] >= $refund_summary['total_sale_items'] && 
                              $refund_summary['total_refunded_quantity'] >= $refund_summary['total_sale_quantity']);
    }

    // Update sale status based on refund completion
    if ($is_complete_refund) {
        $stmt = $pdo->prepare("UPDATE sales SET status = 'refunded' WHERE id = ?");
        $stmt->execute([$sale_id]);
    } else {
        // Keep status as 'completed' for partial refunds
        $stmt = $pdo->prepare("UPDATE sales SET status = 'completed' WHERE id = ?");
        $stmt->execute([$sale_id]);
    }

    // Commit transaction
    $pdo->commit();

    // Send notifications to customer
    $notification_results = [];
    
    // Get notification preferences from request
    $notifications = $input['notifications'] ?? [];
    $send_whatsapp = $notifications['send_whatsapp'] ?? false;
    $send_sms = $notifications['send_sms'] ?? false;
    $send_email = $notifications['send_email'] ?? false;
    
    // Get customer details
    $stmt = $pdo->prepare("SELECT c.name, c.phone, c.email FROM customers c WHERE c.id = ?");
    $stmt->execute([$sale['customer_id']]);
    $customer = $stmt->fetch();
    
    if ($customer) {
        // Get store settings for notifications
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group IN ('sms_settings', 'email_settings', 'whatsapp_settings', 'store_info')");
        $stmt->execute();
        $settings = [];
        while ($row = $stmt->fetch()) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }
        
        // Prepare refund message
        $storeName = $settings['store_name'] ?? 'Our Store';
        $currency = $settings['currency'] ?? 'Ksh';
        
        // Get refund items for message
        $stmt = $pdo->prepare("
            SELECT ri.quantity, ri.condition, p.name as product_name
            FROM refund_items ri
            JOIN products p ON ri.product_id = p.id
            WHERE ri.refund_id = ?
        ");
        $stmt->execute([$refund_id]);
        $refund_items = $stmt->fetchAll();
        
        $itemsList = [];
        foreach ($refund_items as $item) {
            $itemsList[] = $item['quantity'] . 'x ' . $item['product_name'] . ' (' . $item['condition'] . ')';
        }
        
        $refundMessage = "SALES RETURN CONFIRMATION - " . $storeName . ".\n\n";
        $refundMessage .= "Dear " . $customer['name'] . ",\n";
        $refundMessage .= "We have received your Sales return As Follows\n\n";
        $refundMessage .= "Return Details:\n";
        $refundMessage .= "Receipt: " . $sale['invoice_number'] . "\n";
        $refundMessage .= "Items: " . implode(', ', $itemsList) . "\n";
        $refundMessage .= "Date: " . date('M d, Y H:i') . "\n\n";
        $refundMessage .= "Thank you for choosing " . $storeName . "!\n";
        $refundMessage .= "For any questions, please contact us | " . ($settings['store_phone'] ?? 'N/A') . ".";
        
        $emailSubject = "Return Processed - " . $storeName;
        $emailMessage = nl2br($refundMessage);
        
        // Send SMS
        if ($send_sms && ($settings['enable_sms'] ?? '0') === '1') {
            if (!empty($customer['phone'])) {
                require_once '../includes/sms_functions.php';
                try {
                    $smsResult = sendSMS($customer['phone'], $refundMessage);
                    $notification_results['sms'] = $smsResult;
                } catch (Exception $e) {
                    $notification_results['sms'] = ['success' => false, 'error' => $e->getMessage()];
                }
            } else {
                $notification_results['sms'] = ['success' => false, 'error' => 'No phone number available'];
            }
        }
        
        // Send Email
        if ($send_email && ($settings['enable_email'] ?? '0') === '1') {
            if (!empty($customer['email'])) {
                require_once '../includes/email_functions.php';
                try {
                    $emailResult = sendEmail($customer['email'], $emailSubject, $emailMessage);
                    $notification_results['email'] = $emailResult;
                } catch (Exception $e) {
                    $notification_results['email'] = ['success' => false, 'error' => $e->getMessage()];
                }
            } else {
                $notification_results['email'] = ['success' => false, 'error' => 'No email address available'];
            }
        }
        
        // Send WhatsApp
        if ($send_whatsapp && ($settings['enable_whatsapp'] ?? '0') === '1') {
            if (!empty($customer['phone'])) {
                require_once '../includes/whatsapp_functions.php';
                try {
                    $whatsappResult = sendWhatsApp($customer['phone'], $refundMessage);
                    
                    // If using web_link method, include the WhatsApp URL for automatic opening
                    if ($whatsappResult['success'] && isset($whatsappResult['whatsapp_url'])) {
                        $whatsappResult['auto_open_url'] = $whatsappResult['whatsapp_url'];
                    }
                    
                    $notification_results['whatsapp'] = $whatsappResult;
                } catch (Exception $e) {
                    $notification_results['whatsapp'] = ['success' => false, 'error' => $e->getMessage()];
                }
            } else {
                $notification_results['whatsapp'] = ['success' => false, 'error' => 'No phone number available'];
            }
        }
    }

    echo json_encode([
        'success' => true,
        'message' => 'Refund processed successfully',
        'refund_id' => $refund_id,
        'restocked_items' => $restocked_items,
        'payment_method' => $sale['payment_method'],
        'is_credit_sale' => $sale['payment_method'] === 'credit',
        'refund_breakdown' => $refund_breakdown,
        'has_multiple_payments' => count($sale_payments) > 1,
        'is_complete_refund' => $is_complete_refund,
        'refund_summary' => $refund_summary,
        'notifications_sent' => $notification_results
    ]);

} catch (Exception $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    error_log("Refund Error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Error processing refund: ' . $e->getMessage()
    ]);
} 